// =====================================================
// LotoSport v3.3 - Extensão para Loterias Online Caixa

// =====================================================

// Variáveis globais para controle do processo
let processing = false;
let cancelRequested = false;

// =====================================================
// FUNÇÕES UTILITÁRIAS
// =====================================================

async function getCurrentTab() {
    let queryOptions = { active: true, currentWindow: true };
    let [tab] = await chrome.tabs.query(queryOptions);
    return tab;
}

function delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}

function eliminarRepetidos(value, index, self) {
    return self.indexOf(value) === index;
}

function eliminarNumerosRepetidosNoJogo(arrayDezenas) {
    return [...new Set(arrayDezenas)];
}

function getTrevosAleatorios(n) {
    var arr = ['1','2','3','4','5','6'];
    var result = new Array(n),
        len = arr.length,
        taken = new Array(len);
    if (n > len)
        throw new RangeError("getRandom: more elements taken than available");
    while (n--) {
        var x = Math.floor(Math.random() * len);
        result[n] = arr[x in taken ? taken[x] : x];
        taken[x] = --len in taken ? taken[len] : len;
    }
    return result;
}

function getTimeAleatorio(n) {
    var arr = [
        'ABC/RN','ALTOS/PI','AMAZONAS/AM','AMERICA/MG','AMERICA/RN',
        'APARECIDENSE/GO','ATHLETIC CLUB/MG','ATHLETICO/PR','ATLETICO/GO','ATLETICO/MG',
        'AVAI/SC','BAHIA/BA','BAHIA DE FEIRA/BA','BOTAFOGO/PB','BOTAFOGO/RJ',
        'BOTAFOGO/SP','BRAGANTINO/SP','BRASIL DE PELOTAS/RS','BRASILIENSE/DF','BRUSQUE/SC',
        'CAMPINENSE/PB','CASCAVEL/PR','CAXIAS/RS','CEARA/CE','CEILANDIA ESPORTE/DF',
        'CHAPECOENSE/SC','CONFIANCA/SE','CORINTHIANS/SP','CORITIBA/PR','CRB/AL',
        'CRICIUMA/SC','CRUZEIRO/MG','CSA/AL','CUIABA/MT','FERROVIARIA/SP',
        'FERROVIARIO/CE','FIGUEIRENSE/SC','FLAMENGO/RJ','FLORESTA/CE','FLUMINENSE/RJ',
        'FORTALEZA/CE','GOIAS/GO','GREMIO/RS','GUARANI/SP','INTERNACIONAL/RS',
        'ITUANO/SP','JACUIPENSE/BA','JUAZEIRENSE/BA','JUVENTUDE/RS','LONDRINA/PR',
        'MANAUS/AM','MIRASSOL/SP','NAUTICO/PE','NOVA IGUACU/RJ','NOVORIZONTINO/SP',
        'OESTE/SP','OPERARIO/PR','PALMEIRAS/SP','PARANA/PR','PAYSANDU/PA',
        'PONTE PRETA/SP','PORTUGUESA/RJ','POUSO ALEGRE/MG','REMO/PA','RETRO BRASIL/PE',
        'SAMP CORREA/MA','SANTA CRUZ/PE','SANTOS/SP','SAO BERNARDO/SP','SAO JOSE/RS',
        'SAO PAULO/SP','SAORAIMUNDO/RR','SPORT/PE','TOCANTINOPOLIS/TO','TOMBENSE/MG',
        'VASCO DA GAMA/RJ','VILA NOVA/GO','VITORIA/BA','VOLTA REDONDA/RJ','YPIRANGA/RS',
    ];
    var result = new Array(n),
        len = arr.length,
        taken = new Array(len);
    if (n > len)
        throw new RangeError("getRandom: more elements taken than available");
    while (n--) {
        var x = Math.floor(Math.random() * len);
        result[n] = arr[x in taken ? taken[x] : x];
        taken[x] = --len in taken ? taken[len] : len;
    }
    return result[0];
}

function getMesAleatorio(n) {
    var arr = ['0','1','2','3','4','5','6','7','8','9','10','11'];
    var result = new Array(n),
        len = arr.length,
        taken = new Array(len);
    if (n > len)
        throw new RangeError("getRandom: more elements taken than available");
    while (n--) {
        var x = Math.floor(Math.random() * len);
        result[n] = arr[x in taken ? taken[x] : x];
        taken[x] = --len in taken ? taken[len] : len;
    }
    return result[0];
}

function mesNomeParaIndice(nomeMes) {
    var mapa = {
        'Janeiro': '0', 'Fevereiro': '1', 'Março': '2', 'Abril': '3',
        'Maio': '4', 'Junho': '5', 'Julho': '6', 'Agosto': '7',
        'Setembro': '8', 'Outubro': '9', 'Novembro': '10', 'Dezembro': '11'
    };
    return mapa[nomeMes] || null;
}

// =====================================================
// FUNÇÕES DE UI (Progresso / Lock)
// =====================================================

function updateProgress(current, total) {
    var progressBar = document.getElementById('progressBar');
    var percentage = Math.round((current / total) * 100);
    progressBar.style.width = percentage + '%';
    progressBar.textContent = current + '/' + total + ' (' + percentage + '%)';
    $("#jogo_importando").html(current);
    $("#total_jogos").html(total);
}

function toggleProcessingUI(isProcessing) {
    document.getElementById('progressContainer').classList.toggle('active', isProcessing);
    document.getElementById('cancelContainer').style.display = isProcessing ? 'block' : 'none';
    document.getElementById('btn_importar').disabled = isProcessing;
    document.getElementById('modalidade_code').disabled = isProcessing;
    document.getElementById('jogos_box').disabled = isProcessing;
    document.getElementById('btn_limpar_campo').disabled = isProcessing;

    if (isProcessing) {
        $(".contador_jogos").addClass("active");
    } else {
        $(".contador_jogos").removeClass("active");
    }
}

// =====================================================
// FUNÇÕES INJETADAS NO DOM DA CAIXA (executadas via chrome.scripting)
// Funções executadas via chrome.scripting no DOM da Caixa
// =====================================================

// Função injetada: clica nos números do volante (loterias normais)
function injetarCodigo(linha, modalidade, aposta_espelho, mes_escolhido, trevos, time_coracao, teimosinha, qtd_dezenas) {

    document.getElementById('limparvolante').click();

    // Aumenta quantidade de números se necessário
    if (qtd_dezenas != 0) {
        for (var c = 0; c < qtd_dezenas; c++) {
            document.getElementById("aumentarnumero").click();
        }
    }

    // Espelho para Lotomania
    if (modalidade == '3' && aposta_espelho == 'com') {
        document.getElementById('apostaEspelho').click();
    }

    // Mês para Dia de Sorte
    if (modalidade == '5') {
        var meses = [
            "Janeiro", "Fevereiro", "Março", "Abril", "Maio", "Junho",
            "Julho", "Agosto", "Setembro", "Outubro", "Novembro", "Dezembro"
        ];
        var mes_procurado = meses[mes_escolhido];
        var itens = document.querySelectorAll("#carrossel_diadesorte ul li");
        for (var i = 0; i < itens.length; i++) {
            var spanMes = itens[i].querySelector(".nomeMes span");
            if (spanMes && spanMes.textContent.trim() === mes_procurado) {
                itens[i].click();
                break;
            }
        }
    }

    // Time do Coração para Timemania
    if (modalidade == '8') {
        var time_procurado = time_coracao.trim();
        var itens = document.querySelectorAll("#carrossel_timemania ul li");
        for (var i = 0; i < itens.length; i++) {
            var spanTime = itens[i].querySelector("span.nomeTime");
            if (spanTime && spanTime.innerText.trim() === time_procurado) {
                var imgTime = itens[i].querySelector("img");
                if (imgTime) {
                    var event = new MouseEvent("click", {
                        bubbles: true,
                        cancelable: true,
                        view: window
                    });
                    imgTime.dispatchEvent(event);
                    break;
                }
            }
        }
    }

    // Teimosinha
    if (teimosinha != 0) {
        for (var c = 0; c < teimosinha; c++) {
            document.getElementById('aumentarteimosinha').click();
        }
    }

    // Trevos para +Milionária
    if (modalidade == '9') {
        for (var i = 2; i < trevos.length; i++) {
            document.querySelector("#step6 #aumentarnumero").click();
        }
        for (var i = 0; i < trevos.length; i++) {
            var v = document.getElementById('trevo' + trevos[i]);
            if (v != null) v.click();
        }
    }

    // Clica nas dezenas
    var volante = linha.split(' ');

    if (modalidade == '7') {
        // Super Sete: IDs diretos (ex: n1, n8, n15...)
        for (var i = 0; i < volante.length; i++) {
            var v = document.getElementById(volante[i]);
            if (v != null) v.click();
        }
    } else {
        for (var i = 0; i < volante.length; i++) {
            var dz;
            if (modalidade == '9') {
                dz = parseInt(volante[i]);
            } else {
                dz = volante[i];
            }
            var v = document.getElementById('n' + dz);
            if (v != null) v.click();
        }
    }

    document.getElementById('colocarnocarrinho').click();
    return true;
}

// Navegação de loteria no menu da Caixa
function selecionarLoteria(loteria) {
    document.querySelector("#menuPrincipal a#" + loteria).click();
}

// =====================================================
// PARSING DE JOGOS (mantido da LotoSport)
// =====================================================

function parsearJogoNormal(linhaOriginal, modalidade_id) {
    // Separa valor especial via #
    var jogoCompletoArray = linhaOriginal.split(/[#]+/);
    var jogoValorEspecial = jogoCompletoArray.length > 1 ? jogoCompletoArray[1] : '';

    // Separa dezenas por tab, espaço, vírgula, ponto ou traço
    var jogo_completo = jogoCompletoArray[0].split(/[\t ,.\-]+/);

    // Limpa e formata dezenas
    var jogo_completo_formatado = [];
    for (var k = 0; k < jogo_completo.length; k++) {
        var dezena_corrigida = jogo_completo[k].replace(/[^0-9]/g, '');
        if (dezena_corrigida !== '') {
            if (modalidade_id != '7') {
                dezena_corrigida = dezena_corrigida.padStart(2, '0');
            }
            jogo_completo_formatado.push(dezena_corrigida);
        }
    }

    // Elimina números duplicados dentro do mesmo jogo
    jogo_completo_formatado = eliminarNumerosRepetidosNoJogo(jogo_completo_formatado);

    // Lotomania: move '00' para o final
    if (modalidade_id == '3') {
        var index = jogo_completo_formatado.indexOf('00');
        if (index !== -1) {
            jogo_completo_formatado.splice(index, 1);
            jogo_completo_formatado.push('00');
        }
    }

    return {
        dezenas: jogo_completo_formatado,
        valorEspecial: jogoValorEspecial
    };
}

function parsearJogoSuperSete(linhaOriginal) {
    var jogo_completo = linhaOriginal.split(",");
    var jogo_completo_formatado = [];

    for (var k = 0; k < jogo_completo.length; k++) {
        var dezena_corrigir = jogo_completo[k].split('-');
        var dezena_selecionada = parseInt(dezena_corrigir[1]);
        var coluna_selecionada = dezena_corrigir[0].toString().trim();

        var coluna_num = parseInt(coluna_selecionada.replace('C', ''));
        if (coluna_num >= 1 && coluna_num <= 7 && dezena_selecionada >= 0 && dezena_selecionada <= 9) {
            var dezena_corrigida = 'n' + ((dezena_selecionada * 7) + coluna_num).toString();
            jogo_completo_formatado.push(dezena_corrigida);
        }
    }

    // Elimina duplicados
    jogo_completo_formatado = eliminarNumerosRepetidosNoJogo(jogo_completo_formatado);

    return jogo_completo_formatado;
}

function calcularQtdDezenas(modalidade_id, tamanho) {
    switch (modalidade_id) {
        case '1': return tamanho - 6;
        case '2': return tamanho - 15;
        case '3': return 0;
        case '4': return tamanho - 5;
        case '5': return tamanho - 7;
        case '6': return tamanho - 6;
        case '8': return 0;
        case '9': return tamanho - 6;
        default: return 0;
    }
}

function resolverMes(mes_escolhido, jogoValorEspecial) {
    if (mes_escolhido == 'jogo') {
        var idx = mesNomeParaIndice(jogoValorEspecial);
        return idx !== null ? idx : getMesAleatorio(1);
    }
    if (mes_escolhido == 'variados') {
        return getMesAleatorio(1);
    }
    return mes_escolhido;
}

function resolverTime(selecao_time, jogoValorEspecial) {
    if (selecao_time == 'variados') {
        return getTimeAleatorio(1);
    }
    if (selecao_time == 'jogo') {
        return jogoValorEspecial !== '' ? jogoValorEspecial : getTimeAleatorio(1);
    }
    return selecao_time;
}

function resolverTrevos(tipo_selecao_trevos, jogoValorEspecial) {
    if (tipo_selecao_trevos != 'jogo') {
        return getTrevosAleatorios(parseInt(tipo_selecao_trevos));
    }
    if (jogoValorEspecial !== '') {
        return jogoValorEspecial.split(',');
    }
    return getTrevosAleatorios(2);
}

// =====================================================
// ENGINE DE IMPORTAÇÃO (async/await)
// =====================================================

function executarScriptNaAba(tabId, func, args) {
    return new Promise((resolve) => {
        chrome.scripting.executeScript({
            target: { tabId: tabId },
            func: func,
            args: args || []
        }).then((results) => {
            resolve(results);
        }).catch((error) => {
            console.error('Erro ao executar script:', error);
            resolve(null);
        });
    });
}

async function importarTodosJogos(linhas, modalidade_id, tem_teimosinha) {
    var tab = await getCurrentTab();

    if (!tab || !tab.url) {
        alert("Não foi possível acessar a aba atual. Abra o site das Loterias Online da Caixa.");
        return;
    }

    // Se estiver na home, navega para a loteria
    if (tab.url === 'https://www.loteriasonline.caixa.gov.br/silce-web/#/home') {
        var loteria_escolher = document.getElementById('modalidade_code').value;
        await executarScriptNaAba(tab.id, selecionarLoteria, [loteria_escolher]);
        await delay(2000);
    }

    var total = linhas.length;
    updateProgress(0, total);

    var aposta_espelho = null;
    var tipo_selecao_trevos = $("#selecao_qtd_trevos").val();
    var selecao_time = $("#timeCoracao").val();
    var mes_escolhido_dropdown = $("#mesescolhido").val();

    if (modalidade_id == '3') {
        aposta_espelho = $("#apostarespelho").val();
    }

    for (var i = 0; i < total; i++) {
        if (cancelRequested) break;

        var linhaAtual = linhas[i].trim();
        if (linhaAtual === '') continue;

        if (modalidade_id == '7') {
            // Super Sete
            var dezenasSuperSete = parsearJogoSuperSete(linhaAtual);
            if (dezenasSuperSete.length >= 7 && dezenasSuperSete.length <= 21) {
                dezenasSuperSete.sort();
                var qtd_dezenas = dezenasSuperSete.length - 7;
                var jogo_selecionar = dezenasSuperSete.join(' ');

                await executarScriptNaAba(tab.id, injetarCodigo, [
                    jogo_selecionar, modalidade_id, null, null, null, null,
                    tem_teimosinha, qtd_dezenas
                ]);
            }
        } else {
            // Todas as outras loterias
            var parsed = parsearJogoNormal(linhaAtual, modalidade_id);
            var dezenas = parsed.dezenas;
            var valorEspecial = parsed.valorEspecial;

            if (dezenas.length < 5) {
                // Jogo inválido, pula
                continue;
            }

            var qtd_dezenas = calcularQtdDezenas(modalidade_id, dezenas.length);
            var mes_escolhido = null;
            var time_coracao = null;
            var trevos = null;

            // Dia de Sorte: resolver mês
            if (modalidade_id == '5') {
                mes_escolhido = resolverMes(mes_escolhido_dropdown, valorEspecial);
            }

            // Timemania: resolver time
            if (modalidade_id == '8') {
                time_coracao = resolverTime(selecao_time, valorEspecial);
            }

            // +Milionária: resolver trevos
            if (modalidade_id == '9') {
                trevos = resolverTrevos(tipo_selecao_trevos, valorEspecial);
            }

            var jogo_selecionar = dezenas.join(' ');

            await executarScriptNaAba(tab.id, injetarCodigo, [
                jogo_selecionar, modalidade_id, aposta_espelho, mes_escolhido,
                trevos, time_coracao, tem_teimosinha, qtd_dezenas
            ]);
        }

        // Atualiza progresso e remove jogo do textarea
        updateProgress(i + 1, total);

        var jogos_box = document.getElementById("jogos_box");
        var linhasRestantes = jogos_box.value.split('\n');
        linhasRestantes.shift();
        jogos_box.value = linhasRestantes.join('\n');

        // Delay entre jogos (600ms)
        if (i < total - 1) {
            await delay(600);
        }
    }
}

// =====================================================
// SETUP INICIAL
// =====================================================

async function setup() {
    var tab = await getCurrentTab();
    chrome.scripting.executeScript({
        target: { tabId: tab.id },
        files: ['jquery.js'],
    }, () => {
        document.getElementById("jogos_box").value = localStorage.getItem("jogos_salvar") || '';
    });
}

setup();

// =====================================================
// EVENT LISTENERS
// =====================================================

// Mudança de modalidade
var modalidade_alterar = document.getElementById('modalidade_code');

modalidade_alterar.addEventListener('change', async () => {
    var loteria = modalidade_alterar.value;

    // Usa background.js para navegar na página da Caixa
    var tab = await getCurrentTab();
    chrome.runtime.sendMessage({
        action: "executar_script",
        tabId: tab.id,
        value: loteria
    });

    // Reseta opções
    $("#teimosinhavalor option").remove();
    $(".apostarespelho").removeClass("active");
    $(".mesescolhido").removeClass("active");
    $(".trevo_seletor").removeClass("active");
    $(".selecao_qtd_trevos").removeClass("active");
    $(".timeCoracao").removeClass("active");

    switch (loteria) {
        case 'Mega-Sena':
            $("#loteria_id").val('1');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option><option value="1">Teimosinha de 2</option><option value="2">Teimosinha de 4</option><option value="3">Teimosinha de 8</option>');
            break;
        case 'data-jogo-mega-sena-especial':
            $("#loteria_id").val('1');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option>');
            break;
        case 'Lotofácil':
            $("#loteria_id").val('2');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option><option value="1">Teimosinha de 2</option><option value="2">Teimosinha de 3</option><option value="3">Teimosinha de 4</option><option value="4">Teimosinha de 6</option><option value="5">Teimosinha de 8</option><option value="6">Teimosinha de 9</option><option value="7">Teimosinha de 12</option><option value="8">Teimosinha de 18</option><option value="9">Teimosinha de 24</option>');
            break;
        case 'data-jogo-lotofacil-especial':
            $("#loteria_id").val('2');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option>');
            break;
        case 'Lotomania':
            $("#loteria_id").val('3');
            $(".apostarespelho").addClass("active");
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option><option value="1">Teimosinha de 2</option><option value="2">Teimosinha de 4</option><option value="3">Teimosinha de 8</option>');
            break;
        case 'Quina':
            $("#loteria_id").val('4');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option><option value="1">Teimosinha de 3</option><option value="2">Teimosinha de 6</option><option value="3">Teimosinha de 12</option><option value="4">Teimosinha de 18</option><option value="5">Teimosinha de 24</option>');
            break;
        case 'data-jogo-quina-especial':
            $("#loteria_id").val('4');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option>');
            break;
        case 'Dia-de-Sorte':
            $("#loteria_id").val('5');
            $(".mesescolhido").addClass("active");
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option><option value="1">Teimosinha de 3</option><option value="2">Teimosinha de 6</option><option value="3">Teimosinha de 9</option><option value="4">Teimosinha de 12</option>');
            break;
        case 'Dupla-Sena':
            $("#loteria_id").val('6');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option><option value="1">Teimosinha de 2</option><option value="2">Teimosinha de 3</option><option value="3">Teimosinha de 4</option><option value="4">Teimosinha de 6</option><option value="5">Teimosinha de 8</option><option value="6">Teimosinha de 9</option><option value="7">Teimosinha de 12</option>');
            break;
        case 'data-jogo-dupla-sena-especial':
            $("#loteria_id").val('6');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option>');
            break;
        case 'Super-Sete':
            $("#loteria_id").val('7');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option><option value="1">Teimosinha de 3</option><option value="2">Teimosinha de 6</option><option value="3">Teimosinha de 9</option><option value="4">Teimosinha de 12</option>');
            break;
        case 'Timemania':
            $("#loteria_id").val('8');
            $(".timeCoracao").addClass("active");
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option><option value="1">Teimosinha de 3</option><option value="2">Teimosinha de 6</option><option value="3">Teimosinha de 9</option><option value="4">Teimosinha de 12</option>');
            break;
        case 'Mais-Milionaria':
            $("#loteria_id").val('9');
            $(".trevo_seletor").removeClass("active");
            $(".selecao_qtd_trevos").addClass("active");
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option><option value="1">Teimosinha de 2</option><option value="2">Teimosinha de 3</option><option value="3">Teimosinha de 4</option><option value="5">Teimosinha de 5</option>');
            break;
        default:
            $("#loteria_id").val('2');
            $("#teimosinhavalor").append('<option value="0">Sem Teimosinha</option>');
            break;
    }
});

// Salvar jogos no localStorage quando textarea muda
jQuery(document).on('change', '#jogos_box', function(e) {
    var jogos = document.getElementById("jogos_box").value;
    localStorage.setItem("jogos_salvar", jogos);
});

// Botão IMPORTAR
var importar_jogos_btn = document.getElementById('btn_importar');
importar_jogos_btn.addEventListener('click', async () => {
    if (processing) return;

    var jogos_box = document.getElementById("jogos_box");
    var jogos = jogos_box.value;

    if (!jogos || !jogos.trim()) {
        alert("Cole os jogos no campo de texto antes de importar.");
        return;
    }

    var modalidade_id = $("#loteria_id").val();
    if (!modalidade_id) {
        alert("Selecione uma loteria antes de importar.");
        return;
    }

    // Filtra linhas vazias e elimina linhas repetidas
    var linhas = jogos.split('\n');
    var linhasFiltradas = linhas.filter(function (linha) {
        return linha.trim() !== "";
    });
    linhas = linhasFiltradas.filter(eliminarRepetidos);
    jogos_box.value = linhas.join('\n');

    if (linhas.length === 0) {
        alert("Nenhum jogo válido encontrado.");
        return;
    }

    var tem_teimosinha = $("#teimosinhavalor").val();

    // Ativa UI de processamento
    processing = true;
    cancelRequested = false;
    toggleProcessingUI(true);

    try {
        await importarTodosJogos(linhas, modalidade_id, tem_teimosinha);

        // Força progresso final e aguarda a transição CSS (300ms) completar antes do alert
        updateProgress(linhas.length, linhas.length);
        await delay(400);

        if (cancelRequested) {
            alert("Importação cancelada pelo usuário.");
        } else {
            alert("Importação concluída! " + linhas.length + " jogo(s) processado(s).");
        }
    } catch (error) {
        console.error("Erro durante a importação:", error);
        alert("Ocorreu um erro durante a importação.");
    } finally {
        processing = false;
        toggleProcessingUI(false);
    }
});

// Botão CANCELAR
document.getElementById('btn_cancelar').addEventListener('click', function() {
    cancelRequested = true;
});

// Botão LIMPAR
jQuery(document).on('click', '#btn_limpar_campo', function(e) {
    $("#jogos_box").val("");
    $("#total_jogos").html("0");
    $("#jogo_importando").html("0");
    $(".contador_jogos").removeClass("active");
    document.getElementById('progressContainer').classList.remove('active');
    document.getElementById('cancelContainer').style.display = 'none';
    localStorage.removeItem("jogos_salvar");
});
